/*
 * @brief Secondary loader I2C host interface handler
 *
 * @note
 * Copyright(C) NXP Semiconductors, 2014
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */

#include "chip.h"
#include "sl_common.h"
#include "fsl_i2c.h"

typedef void (*I2CSlaveXferStart)(uint8_t addr);
typedef uint8_t (*I2CSlaveXferSend)(uint8_t *data);
typedef uint8_t (*I2CSlaveXferRecv)(uint8_t data);
typedef void (*I2CSlaveXferDone)(void);

typedef struct {
    I2CSlaveXferStart slaveStart;      /*!< Called when an matching I2C slave address is received */
    I2CSlaveXferSend slaveSend;        /*!< Called when a byte is needed to send to master */
    I2CSlaveXferRecv slaveRecv;        /*!< Called when a byte is received from master */
    I2CSlaveXferDone slaveDone;        /*!< Called when a slave transfer is complete */
} I2CS_XFER_T;

/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/
static I2CS_XFER_T i2csCallBacks[3];

static I2C_Type *pI2CArray[1] = {
    I2C0
};

static int buffSendIndex;
extern int buffXferPending;

/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/

/*****************************************************************************
 * Private functions
 ****************************************************************************/

/* Handler for slave start callback */
static void processSlaveTransferStart(uint8_t addr)
{
    Hostif_DeAssertIRQ();

    /* Setup transfer at the start */
    buffSendIndex = 0;

    /* Receive buffer index always reset to 0 on asssert event */
    recvBuff.bytes = 0;
}

/* Handler for slave send callback */
static uint8_t processSlaveTransferSend(uint8_t *data)
{
    if (buffSendIndex < sendBuff.bytes) {
        *data = sendBuff.buffer[buffSendIndex];
        buffSendIndex++;
    }
    else {
        /* Can't NAK on a send with master controlling transfer,
           but we'll return NAK for completion. It isn't used. */
        *data = CFG_BOOT_WHO_AM_I;    /* send WHOAMI identifier as marker for receiver to know no data is pending*/
        return I2C_SLVCTL_SLVNACK_MASK;
    }

    return 0;
}

/* Handler for slave receive callback */
static uint8_t processSlaveTransferRecv(uint8_t data)
{
    recvBuff.buffer[recvBuff.bytes] = data;
    recvBuff.bytes++;

    /* To be consistent with SPI new protocol doesn't require sending NAK on last byte received */
    return 0;
}

/* Handler for slave transfer complete callback */
static void processSlaveTransferDone(void)
{
    buffXferPending = 1;
}


static void processSlaveTransferDone_0(void)
{
    slIfConfig.ifSel = SL_I2C0;
    processSlaveTransferDone();
}

static void processSlaveTransferDone_1(void)
{
    slIfConfig.ifSel = SL_I2C1;
    processSlaveTransferDone();
}

static void processSlaveTransferDone_2(void)
{
    slIfConfig.ifSel = SL_I2C2;
    processSlaveTransferDone();
}

/*****************************************************************************
 * Public functions
 ****************************************************************************/

/* Sets up pinmuxing for an I2C interface */
void setupMuxingI2C(int i2cNum)
{
    if(i2cNum == 0){
        
        /* Provide main_clk as function clock to I2C0 */
        SYSCON->FCLKSEL[5] = SYSCON_FCLKSEL_SEL(1);
        SYSCON->SYSAHBCLKCTRL0 |= ( SYSCON_SYSAHBCLKCTRL0_SWM_MASK \
                               | SYSCON_SYSAHBCLKCTRL0_GPIO0_MASK \
                               | SYSCON_SYSAHBCLKCTRL0_I2C0_MASK \
                               | SYSCON_SYSAHBCLKCTRL0_IOCON_MASK);
        
        RESET_PeripheralReset(kI2C0_RST_N_SHIFT_RSTn);
        
        /* I2C FASH MODE */
        I2C0->MSTTIME &= 0XFFFFFF81;
        I2C0->CLKDIV = 5;
                
    } else if(i2cNum == 1){
    }
            
}

typedef void *I2C_HANDLE_T;

void setupInterfaceI2C(int i2cNum)
{
    /* Enable I2C clock and reset I2C peripheral */
        
    /* Set a single 7-bit I2C address, only 7-bit addressing is supported */
    I2C_SlaveSetAddress(I2C0, kI2C_SlaveAddressRegister0, SL_I2C_SLAVE_ADDR_0,0);
    I2C_SlaveSetAddress(I2C0, kI2C_SlaveAddressRegister1, SL_I2C_SLAVE_ADDR_LENOVO,0);
    I2C_SlaveSetAddress(I2C0, kI2C_SlaveAddressRegister2, SL_I2C_SLAVE_ADDR_2,0);
    
    /* Clear interrupt status and enable slave interrupts */
    I2C_SlaveClearStatusFlags(I2C0, I2C_STAT_SLVDESEL_MASK);

    /* Interrupt enabled, but NVIC is not enabled in this app! */
    I2C_EnableInterrupts(I2C0, I2C_INTSTAT_SLVPENDING_MASK | I2C_INTSTAT_SLVDESEL_MASK);
}

void startHostIfI2C(int i2cNum)
{
    int i;

    /* I2C slavecallback function list */
    for (i = 0; i < 3; i++) {
        i2csCallBacks[i].slaveStart = &processSlaveTransferStart;
        i2csCallBacks[i].slaveSend  = &processSlaveTransferSend;
        i2csCallBacks[i].slaveRecv  = &processSlaveTransferRecv;
    }
    i2csCallBacks[0].slaveDone = &processSlaveTransferDone_0;
    i2csCallBacks[1].slaveDone = &processSlaveTransferDone_1;
    i2csCallBacks[2].slaveDone = &processSlaveTransferDone_2;

    /* Enable I2C slave interface */
    I2C_SlaveEnable(pI2CArray[i2cNum], true);
    /* This I2C interface is now active */
}

/* Slave transfer state change handler */

uint32_t Chip_I2CS_XferHandler(I2C_Type *pI2C, const I2CS_XFER_T *xfers)
{
    uint32_t done = 0;

    uint8_t data;
    uint32_t state;
    uint8_t index;

    /* transfer complete? */
    if ((I2C_GetEnabledInterrupts(I2C0) & I2C_STAT_SLVDESEL_MASK) != 0) {
        I2C_SlaveClearStatusFlags(pI2C, I2C_STAT_SLVDESEL_MASK);
        xfers->slaveDone();
    }
    else {
        /* Determine the current I2C slave state */
        state = (I2C0->STAT & I2C_STAT_SLVSTATE_MASK) >> 9;

        switch (state) {
        case I2C_STAT_SLVST_ADDR:        /* Slave address received */
            /* Get slave address that needs servicing */
             index = (pI2C->STAT & I2C_STAT_SLVIDX_MASK) >> 12;
             data = (pI2C->SLVADR[index] >> 1) & 0x7F;
        
            /* Call address callback */
            xfers->slaveStart(data);
            break;

        case I2C_STAT_SLVST_RX:        /* Data byte received */
            /* Get received data */
            I2C_SlaveReadBlocking(pI2C, &data, 1);
            done = xfers->slaveRecv(data);
            break;

        case I2C_STAT_SLVST_TX:        /* Get byte that needs to be sent */
            /* Get data to send */
            done = xfers->slaveSend(&data);
            I2C_SlaveWriteBlocking(pI2C, &data, 1);
            break;
        }
    }

    if (done == 0) {
        pI2C->SLVCTL = I2C_SLVCTL_SLVCONTINUE_MASK;
    }
    else {
        pI2C->SLVCTL = I2C_SLVCTL_SLVNACK_MASK;
    }

    return done;
}

/* I2C host interface processing loop */
void loopHostIfI2C(int i2cNum)
{
    uint32_t state = I2C_GetEnabledInterrupts(I2C0);
    /* I2C slave related interrupt */
    while (state & (I2C_STAT_SLVPENDING_MASK | I2C_STAT_SLVDESEL_MASK)) {
        Chip_I2CS_XferHandler(pI2CArray[i2cNum], &i2csCallBacks[i2cNum]);
        
        /* Update state */
        state = I2C_GetEnabledInterrupts(I2C0);
    }
}

/* Shut down a I2C slave interface */
void shutdownInterfaceI2C(int i2cNum)
{
    I2C_SlaveSetAddress(I2C0, kI2C_SlaveAddressRegister0, SL_I2C_SLAVE_ADDR_0,1);
    I2C_SlaveSetAddress(I2C0, kI2C_SlaveAddressRegister1, SL_I2C_SLAVE_ADDR_LENOVO,1);
    I2C_SlaveSetAddress(I2C0, kI2C_SlaveAddressRegister2, SL_I2C_SLAVE_ADDR_2,1);

    I2C_SlaveEnable(pI2CArray[i2cNum], false);
    CLOCK_DisableClock(kCLOCK_I2c0);
    I2C_SlaveDeinit(I2C0);
}

/* I2C interrupts are only used in host interface detection */
void I2C0_IRQHandler(void)
{
    loopHostIfI2C(0);
}



